#ifndef V3D_BASICS_MATRIXPROJ_TPL_H_INCLUDED
#define V3D_BASICS_MATRIXPROJ_TPL_H_INCLUDED

#include "VectorTpl.h"
#include "MatrixTransfTpl.h"

namespace V3D { 


template< typename REAL >
class MatrixProjTpl          // Matrice de projection homogene (4 * 4)
{
	
public:

	enum
	{
		M11 = (4 * 0 + 0),
		M12	= (4 * 0 + 1),
		M13	= (4 * 0 + 2),
		M14	= (4 * 0 + 3),
		
		M21	= (4 * 1 + 0),
		M22	= (4 * 1 + 1),
		M23	= (4 * 1 + 2),
		M24	= (4 * 1 + 3),
		
		M31	= (4 * 2 + 0),
		M32	= (4 * 2 + 1),
		M33	= (4 * 2 + 2),
		M34	= (4 * 2 + 3),
		
		M41	= (4 * 3 + 0),
		M42	= (4 * 3 + 1),
		M43	= (4 * 3 + 2),
		M44	= (4 * 3 + 3)
	};

	
	
	// ATTENTION : Pas de virtuel dans la classe. Ne doit pas etre derivee.
	MatrixProjTpl() {}
	explicit inline MatrixProjTpl( const MatrixTransfTpl<REAL>& mtx);


	inline Vector4Tpl<REAL> operator * (const Vector4Tpl<REAL>& vec) const;
	inline Vector3Tpl<REAL> operator * (const Vector3Tpl<REAL>& vec) const;
	

	template <typename REALOTHER>
		void TransformDirection( Vector3Tpl<REALOTHER>& vc) const
		{
			vc.Set( REAL(m[M11] * vc.x + m[M12] * vc.y + m[M13] * vc.z),
			        REAL(m[M21] * vc.x + m[M22] * vc.y + m[M23] * vc.z),
			        REAL(m[M31] * vc.x + m[M32] * vc.y + m[M33] * vc.z) );
		}



	// Operateurs de multiplication droite de matrices de transformation
	MatrixProjTpl<REAL>& operator *= (const MatrixTransfTpl<REAL>& mat);	
	MatrixProjTpl<REAL>& operator *= (const MatrixProjTpl<REAL>& mat);	



	void SetColumns( const Vector4Tpl<REAL>& v1, const Vector4Tpl<REAL>& v2, const Vector4Tpl<REAL>& v3, const Vector4Tpl<REAL>& v4);
	void GetColumns( Vector4Tpl<REAL>& v1, Vector4Tpl<REAL>& v2, Vector4Tpl<REAL>& v3, Vector4Tpl<REAL>& v4) const;
	void GetRows( Vector4Tpl<REAL>& v1, Vector4Tpl<REAL>& v2, Vector4Tpl<REAL>& v3, Vector4Tpl<REAL>& v4) const;

	void LoadIdentity();
	void Invert();
	void Transpose();


	void GetOGLMatrix( float pafMatEntries[16] ) const;
	void SetFromOGLMatrix( float pafMatEntries[16] );


protected:
	REAL   m[4*4];         // valeurs contenues dans la matrice 4*4

};

//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////


template< typename REAL >
MatrixProjTpl<REAL>::MatrixProjTpl( const MatrixTransfTpl<REAL>& mtx)
{
	const Vector3f& vcColX = mtx.GetColumnX();
	const Vector3f& vcColY = mtx.GetColumnY();
	const Vector3f& vcColZ = mtx.GetColumnZ();
	const Vector3D& vcColTr = mtx.GetTranslation();
	
	m[M11] = vcColX.x; m[M12] = vcColY.x; m[M13] = vcColZ.x; m[M14] = vcColTr.x; 
	m[M21] = vcColX.y; m[M22] = vcColY.y; m[M23] = vcColZ.y; m[M24] = vcColTr.y; 
	m[M31] = vcColX.z; m[M32] = vcColY.z; m[M33] = vcColZ.z; m[M34] = vcColTr.z; 
	m[M41] = 0.f;      m[M42] = 0.f     ; m[M43] = 0.f     ; m[M44] = 1.f; 
}


template< typename REAL >
inline Vector4Tpl<REAL> MatrixProjTpl<REAL>::operator * (const Vector4Tpl<REAL>& vec) const
{
	REAL newx = m[M11] * vec.x + m[M12] * vec.y + m[M13] * vec.z + m[M14]*vec.w;
	REAL newy = m[M21] * vec.x + m[M22] * vec.y + m[M23] * vec.z + m[M24]*vec.w;
	REAL newz = m[M31] * vec.x + m[M32] * vec.y + m[M33] * vec.z + m[M34]*vec.w;
	REAL neww = m[M41] * vec.x + m[M42] * vec.y + m[M43] * vec.z + m[M44]*vec.w;
	return Vector4Tpl<REAL>(newx, newy, newz, neww);
}

template< typename REAL >
inline Vector3Tpl<REAL> MatrixProjTpl<REAL>::operator * (const Vector3Tpl<REAL>& vec) const
{
	REAL newx = m[M11] * vec.x + m[M12] * vec.y + m[M13] * vec.z + m[M14];
	REAL newy = m[M21] * vec.x + m[M22] * vec.y + m[M23] * vec.z + m[M24];
	REAL newz = m[M31] * vec.x + m[M32] * vec.y + m[M33] * vec.z + m[M34];
	REAL neww = m[M41] * vec.x + m[M42] * vec.y + m[M43] * vec.z + m[M44];
	REAL newwinv = REAL(1.0 / neww);
	return Vector3Tpl<REAL>(newx*newwinv, newy*newwinv, newz*newwinv);
}



template< typename REAL >
void MatrixProjTpl<REAL>::SetColumns( const Vector4Tpl<REAL>& v1, const Vector4Tpl<REAL>& v2, const Vector4Tpl<REAL>& v3, const Vector4Tpl<REAL>& v4)
{
	m[M11] = v1.x; m[M21] = v1.y; m[M31] = v1.z; m[M41] = v1.w;
	m[M12] = v2.x; m[M22] = v2.y; m[M32] = v2.z; m[M42] = v2.w;
	m[M13] = v3.x; m[M23] = v3.y; m[M33] = v3.z; m[M43] = v3.w;
	m[M14] = v4.x; m[M24] = v4.y; m[M34] = v4.z; m[M44] = v4.w;
}


template< typename REAL >
void MatrixProjTpl<REAL>::GetColumns( Vector4Tpl<REAL>& v1, Vector4Tpl<REAL>& v2, Vector4Tpl<REAL>& v3, Vector4Tpl<REAL>& v4) const 
{
	v1.Set(m[M11], m[M21], m[M31], m[M41]);
	v2.Set(m[M12], m[M22], m[M32], m[M42]);
	v3.Set(m[M13], m[M23], m[M33], m[M43]);
	v4.Set(m[M14], m[M24], m[M34], m[M44]);
}

template< typename REAL >
void MatrixProjTpl<REAL>::GetRows( Vector4Tpl<REAL>& v1, Vector4Tpl<REAL>& v2, Vector4Tpl<REAL>& v3, Vector4Tpl<REAL>& v4) const
{
	v1.Set( m[M11], m[M12], m[M13], m[M14]);
	v2.Set( m[M21], m[M22], m[M23], m[M24]);
	v3.Set( m[M31], m[M32], m[M33], m[M34]);
	v4.Set( m[M41], m[M42], m[M43], m[M44]);
}


template< typename REAL >
void MatrixProjTpl<REAL>::LoadIdentity()
{
	memset((void*) m, 0, sizeof(m));
	m[M11] = m[M22] = m[M33] = m[M44] = 1.f;
}



template< typename REAL >
void MatrixProjTpl<REAL>::GetOGLMatrix( float pafMatEntries[16] ) const
{
	pafMatEntries[0]  = m[M11]; pafMatEntries[1]  = m[M21]; pafMatEntries[2]  = m[M31]; pafMatEntries[3]  = m[M41];
	pafMatEntries[4]  = m[M12]; pafMatEntries[5]  = m[M22]; pafMatEntries[6]  = m[M32]; pafMatEntries[7]  = m[M42];
	pafMatEntries[8]  = m[M13]; pafMatEntries[9]  = m[M23]; pafMatEntries[10] = m[M33]; pafMatEntries[11] = m[M43];
	pafMatEntries[12] = m[M14]; pafMatEntries[13] = m[M24]; pafMatEntries[14] = m[M34]; pafMatEntries[15] = m[M44];
}

template< typename REAL >
void MatrixProjTpl<REAL>::SetFromOGLMatrix( float pafMatEntries[16] )
{
	m[M11] = pafMatEntries[0] ; m[M21] = pafMatEntries[1] ; m[M31] = pafMatEntries[2] ; m[M41] = pafMatEntries[3] ;
	m[M12] = pafMatEntries[4] ; m[M22] = pafMatEntries[5] ; m[M32] = pafMatEntries[6] ; m[M42] = pafMatEntries[7] ;
	m[M13] = pafMatEntries[8] ; m[M23] = pafMatEntries[9] ; m[M33] = pafMatEntries[10]; m[M43] = pafMatEntries[11];
	m[M14] = pafMatEntries[12]; m[M24] = pafMatEntries[13]; m[M34] = pafMatEntries[14]; m[M44] = pafMatEntries[15];
}




template< typename REAL >
MatrixProjTpl<REAL>& MatrixProjTpl<REAL>::operator *= (const MatrixProjTpl<REAL>& mat)	// Multiplication droite
{
	Vector4Tpl<REAL> vr1, vr2, vr3, vr4;
	GetRows( vr1, vr2, vr3, vr4);

	Vector4Tpl<REAL> vc1, vc2, vc3, vc4;
	mat.GetColumns( vc1, vc2, vc3, vc4);
	
	// | m11  m12  m13  m14 |   | n11  n12  n13  n14 |
	// | m21  m22  m23  m24 | * | n21  n22  n23  n24 |
	// | m31  m32  m33  m34 |   | n31  n32  n33  n34 |
	// | m41  m42  m43  m44 |   | n41  n42  n43  n44 |

	m[M11] = vr1 * vc1;
	m[M12] = vr1 * vc2;
	m[M13] = vr1 * vc3;
	m[M14] = vr1 * vc4;

	m[M21] = vr2 * vc1;
	m[M22] = vr2 * vc2;
	m[M23] = vr2 * vc3;
	m[M24] = vr2 * vc4;

	m[M31] = vr3 * vc1;
	m[M32] = vr3 * vc2;
	m[M33] = vr3 * vc3;
	m[M34] = vr3 * vc4;

	m[M41] = vr4 * vc1;
	m[M42] = vr4 * vc2;
	m[M43] = vr4 * vc3;
	m[M44] = vr4 * vc4;

	return *this;
}


template< typename REAL >
MatrixProjTpl<REAL>& MatrixProjTpl<REAL>::operator *= (const MatrixTransfTpl<REAL>& mat) // Multiplication droite
{
	Vector4Tpl<REAL>   vr1, vr2, vr3, vr4;
	GetRows( vr1, vr2, vr3, vr4);

	Vector4Tpl<REAL>   vc1, vc2, vc3, vc4;
	mat.GetColumns( vc1, vc2, vc3, vc4);


	// | m11  m12  m13  m14 |   | n11  n12  n13  n14 |
	// | m21  m22  m23  m24 | * | n21  n22  n23  n24 |
	// | m31  m32  m33  m34 |   | n31  n32  n33  n34 |
	// | m41  m42  m43  m44 |   |  0    0    0    1  |
	
	m[M11] = vr1 * vc1;
	m[M12] = vr1 * vc2;
	m[M13] = vr1 * vc3;
	m[M14] = vr1 * vc4;

	m[M21] = vr2 * vc1;
	m[M22] = vr2 * vc2;
	m[M23] = vr2 * vc3;
	m[M24] = vr2 * vc4;

	m[M31] = vr3 * vc1;
	m[M32] = vr3 * vc2;
	m[M33] = vr3 * vc3;
	m[M34] = vr3 * vc4;

	m[M41] = vr4 * vc1;
	m[M42] = vr4 * vc2;
	m[M43] = vr4 * vc3;
	m[M44] = vr4 * vc4;
	return *this;
}


template< typename REAL >
void MatrixProjTpl<REAL>::Transpose()
{
	std::swap( m[M21], m[M12] );
	std::swap( m[M31], m[M13] );
	std::swap( m[M41], m[M14] );

	std::swap( m[M32], m[M23] );
	std::swap( m[M42], m[M24] );

	std::swap( m[M43], m[M34] );
}



template< typename REAL >
void MatrixProjTpl<REAL>::Invert()
{
    MatrixProjTpl mtxSrc = *this;
	
	REAL fA0 = mtxSrc.m[M11]*mtxSrc.m[M22] - mtxSrc.m[M12]*mtxSrc.m[M21];
    REAL fA1 = mtxSrc.m[M11]*mtxSrc.m[M23] - mtxSrc.m[M13]*mtxSrc.m[M21];
    REAL fA2 = mtxSrc.m[M11]*mtxSrc.m[M24] - mtxSrc.m[M14]*mtxSrc.m[M21];
    REAL fA3 = mtxSrc.m[M12]*mtxSrc.m[M23] - mtxSrc.m[M13]*mtxSrc.m[M22];
    REAL fA4 = mtxSrc.m[M12]*mtxSrc.m[M24] - mtxSrc.m[M14]*mtxSrc.m[M22];
    REAL fA5 = mtxSrc.m[M13]*mtxSrc.m[M24] - mtxSrc.m[M14]*mtxSrc.m[M23];
    REAL fB0 = mtxSrc.m[M31]*mtxSrc.m[M42] - mtxSrc.m[M32]*mtxSrc.m[M41];
    REAL fB1 = mtxSrc.m[M31]*mtxSrc.m[M43] - mtxSrc.m[M33]*mtxSrc.m[M41];
    REAL fB2 = mtxSrc.m[M31]*mtxSrc.m[M44] - mtxSrc.m[M34]*mtxSrc.m[M41];
    REAL fB3 = mtxSrc.m[M32]*mtxSrc.m[M43] - mtxSrc.m[M33]*mtxSrc.m[M42];
    REAL fB4 = mtxSrc.m[M32]*mtxSrc.m[M44] - mtxSrc.m[M34]*mtxSrc.m[M42];
    REAL fB5 = mtxSrc.m[M33]*mtxSrc.m[M44] - mtxSrc.m[M34]*mtxSrc.m[M43];

    REAL fDet = fA0*fB5-fA1*fB4+fA2*fB3+fA3*fB2-fA4*fB1+fA5*fB0;
	assert( fabs(fDet) > 1E-5 );

    REAL fInvDet = REAL( 1.0 / fDet );

    m[M11] = fInvDet * (  mtxSrc.m[M22]*fB5 - mtxSrc.m[M23]*fB4 + mtxSrc.m[M24]*fB3);
    m[M21] = fInvDet * (- mtxSrc.m[M21]*fB5 + mtxSrc.m[M23]*fB2 - mtxSrc.m[M24]*fB1);
    m[M31] = fInvDet * (  mtxSrc.m[M21]*fB4 - mtxSrc.m[M22]*fB2 + mtxSrc.m[M24]*fB0);
    m[M41] = fInvDet * (- mtxSrc.m[M21]*fB3 + mtxSrc.m[M22]*fB1 - mtxSrc.m[M23]*fB0);
    m[M12] = fInvDet * (- mtxSrc.m[M12]*fB5 + mtxSrc.m[M13]*fB4 - mtxSrc.m[M14]*fB3);
    m[M22] = fInvDet * (  mtxSrc.m[M11]*fB5 - mtxSrc.m[M13]*fB2 + mtxSrc.m[M14]*fB1);
    m[M32] = fInvDet * (- mtxSrc.m[M11]*fB4 + mtxSrc.m[M12]*fB2 - mtxSrc.m[M14]*fB0);
    m[M42] = fInvDet * (  mtxSrc.m[M11]*fB3 - mtxSrc.m[M12]*fB1 + mtxSrc.m[M13]*fB0);
    m[M13] = fInvDet * (  mtxSrc.m[M42]*fA5 - mtxSrc.m[M43]*fA4 + mtxSrc.m[M44]*fA3);
    m[M23] = fInvDet * (- mtxSrc.m[M41]*fA5 + mtxSrc.m[M43]*fA2 - mtxSrc.m[M44]*fA1);
    m[M33] = fInvDet * (  mtxSrc.m[M41]*fA4 - mtxSrc.m[M42]*fA2 + mtxSrc.m[M44]*fA0);
    m[M43] = fInvDet * (- mtxSrc.m[M41]*fA3 + mtxSrc.m[M42]*fA1 - mtxSrc.m[M43]*fA0);
    m[M14] = fInvDet * (- mtxSrc.m[M32]*fA5 + mtxSrc.m[M33]*fA4 - mtxSrc.m[M34]*fA3);
    m[M24] = fInvDet * (  mtxSrc.m[M31]*fA5 - mtxSrc.m[M33]*fA2 + mtxSrc.m[M34]*fA1);
    m[M34] = fInvDet * (- mtxSrc.m[M31]*fA4 + mtxSrc.m[M32]*fA2 - mtxSrc.m[M34]*fA0);
    m[M44] = fInvDet * (  mtxSrc.m[M31]*fA3 - mtxSrc.m[M32]*fA1 + mtxSrc.m[M33]*fA0);
}





} // namespaces


#endif		//#ifndef V3D_BASICS_MATRIXPROJ_TPL_H_INCLUDED

